#include "vector2.h"
#include <cmath>

/** @return true if the given x, y coordinate mathes this Vector2's data */
bool Vector2::is(float x, float y) const
{
	return this->x == x && this->y == y;
}

/** @return true if the given x, y coordinate mathes this Vector2's data */
bool Vector2::is(Vector2 xy) const
{
	return xy.x == x && xy.y == y;
}

/** @return true if the Vector2 is within the given rectangular boundary */
bool Vector2::isWithin(float minx, float miny, float maxx, float maxy)
{
	return x >= minx && y >= miny && x < maxx && y < maxy;
}

/** re-initialize */
void Vector2::init(float a_x, float a_y)
{
	x = a_x;
	y = a_y;
}

/** @return pythagorean distance from the origin */
float Vector2::magnitude()
{
	return std::sqrt(x*x+y*y);
}

Vector2 Vector2::operator+(Vector2 const & v) const { return Vector2(x+v.x,   y+v.y); }
Vector2 Vector2::operator-(Vector2 const & v) const { return Vector2(x-v.x,   y-v.y); }
Vector2 Vector2::operator*(float value) const   { return Vector2(x*value, y*value); }
Vector2 Vector2::operator/(float value) const   { return Vector2(x/value, y/value); }
Vector2 & Vector2::operator+=(Vector2& v)  { x += v.x;   y += v.y;   return *this; }
Vector2 & Vector2::operator-=(Vector2& v)  { x -= v.x;   y -= v.y;   return *this; }
Vector2 & Vector2::operator*=(float value) { x *= value; y *= value; return *this; }
Vector2 & Vector2::operator/=(float value) { x /= value; y /= value; return *this; }

#include "stdlib.h"
float randInUnitInterval()
{
	const int RANDMAX = 32768;
	return (rand() % RANDMAX)/((float)RANDMAX);
}

/** @return a random Vector2 within the specified boundary */
Vector2 Vector2::random(Vector2 min, Vector2 max)
{
	float w = max.x-min.x, h = max.y-min.y;
	return Vector2(
		randInUnitInterval() * w + min.x,
		randInUnitInterval() * h + min.y);
}

/**
 * @param A,B line 1
 * @param C,D line 2
 * @param point __OUT to the intersection of line AB and CD
 * @param dist __OUT the distance along line AB to the intersection
 * @return true if intersection occurs between the lines
 */
bool Vector2::lineIntersection(const Vector2 & A, const Vector2 & B, 
									const Vector2 & C, const Vector2 & D, 
									float & dist, Vector2 & point)
{
	float rTop = (A.y-C.y)*(D.x-C.x)-(A.x-C.x)*(D.y-C.y);
	float rBot = (B.x-A.x)*(D.y-C.y)-(B.y-A.y)*(D.x-C.x);
	float sTop = (A.y-C.y)*(B.x-A.x)-(A.x-C.x)*(B.y-A.y);
	float sBot = (B.x-A.x)*(D.y-C.y)-(B.y-A.y)*(D.x-C.x);
	if ( (rBot == 0) || (sBot == 0))
	{
		//lines are parallel
		return false;
	}
	float r = rTop/rBot;
	float s = sTop/sBot;
	Vector2 delta = B - A;
	dist = delta.magnitude() * r;
	point = A + ( delta * r);
	return ( (r > 0) && (r < 1) && (s > 0) && (s < 1) );
}

/**
 * @return positive if v2 is clockwise of this vector
 * (assume Y points down, X to right)
 */
float Vector2::sign(const Vector2 & v) const
{
	return (x*v.y) - (y*v.x);
}

/** @return true if this point is inside the given triangle */
bool Vector2::isInsideTriangle(Vector2 const & a, Vector2 const & b, 
							   Vector2 const & c) const
{
	float signab = (*this-a).sign(b-a),
	      signbc = (*this-b).sign(c-b),
	      signac = (*this-c).sign(a-c);
	return(((signab>=0) == (signbc>=0)) && ((signbc>=0) == (signac>=0)))
		||(((signab<=0) == (signbc<=0)) && ((signbc<=0) == (signac<=0)));
}
